# 📚 Документация по настройке Zellij + Alacritty + WSL

## 🎯 Обзор

Данная документация описывает настроенную систему управления терминальными сессиями с использованием:
- **Alacritty** - быстрый GPU-ускоренный эмулятор терминала (Windows 11)
- **WSL 2** (Ubuntu) - Windows Subsystem for Linux
- **Zellij** - современный terminal multiplexer (версия 0.43.1)
- **Zsh** - оболочка командной строки

## 🏗️ Архитектура и принципы работы

### Цепочка запуска

```
Windows 11
    └─ Alacritty (терминальный эмулятор)
        └─ WSL 2 (Ubuntu)
            └─ Zsh (shell)
                └─ Zellij (terminal multiplexer)
                    └─ Workspace layouts с панелями
```

### Принципы

1. **Автоматический запуск**: При открытии Alacritty автоматически запускается WSL с Zsh, который предлагает выбор workspace layout
2. **Персистентность**: Layout файлы позволяют воссоздать структуру рабочего пространства после перезагрузки
3. **Гибкость**: Несколько предустановленных layout на выбор (40/60, 50/50)
4. **Интеграция**: Neovim автоматически открывается в нужной директории, htop для мониторинга

## 📁 Структура файлов

### WSL (Ubuntu) - `/home/mxm/`

```
~/
├── .config/
│   └── zellij/
│       └── layouts/
│           ├── workspacePrjSnabjenie.kdl  # Layout 40/60
│           └── my-workspace.kdl            # Layout 50/50
├── .zshrc                                  # Конфигурация Zsh с автозапуском
├── start-zellij-choose.sh                 # Скрипт выбора layout
├── start-prj-snabjenie.sh                 # Прямой запуск layout 40/60
└── start-simple.sh                        # Прямой запуск layout 50/50
```

### Windows 11

```
C:\Users\z6364\AppData\Roaming\alacritty\
├── alacritty.toml                # Основная конфигурация
└── alacritty.toml.backup         # Резервная копия
```

## 📋 Конфигурационные файлы

### 1. Layout: workspacePrjSnabjenie.kdl (40% левая / 60% правая)

**Путь**: `~/.config/zellij/layouts/workspacePrjSnabjenie.kdl`

```kdl
layout {
    default_tab_template {
        pane size=1 borderless=true {
            plugin location="zellij:tab-bar"
        }
        children
        pane size=2 borderless=true {
            plugin location="zellij:status-bar"
        }
    }

    tab name="ProjectSnabjenie" focus=true {
        pane split_direction="vertical" {
            pane split_direction="horizontal" size="40%" {
                pane {
                    cwd "/home/mxm/.config"
                }
                pane size="40%" {
                    command "htop"
                }
            }
            pane split_direction="horizontal" size="60%" {
                pane size="70%" {
                    cwd "/mnt/c/Project/ProjectSnabjenie"
                    command "zsh"
                    args "-i" "-c" "n ."
                }
                pane size="30%" {
                    cwd "/mnt/c/Project/ProjectSnabjenie"
                }
            }
        }
    }

    tab name="Config" {
        pane {
            cwd "/home/mxm/.config"
        }
    }
}
```

**Структура**:
- **Tab 1: "ProjectSnabjenie"** (фокус по умолчанию)
  - Левая сторона (40%):
    - Верх (60%): Терминал в `~/.config`
    - Низ (40%): `htop` - мониторинг системы
  - Правая сторона (60%):
    - Верх (70%): Neovim (алиас `n`) в проекте
    - Низ (30%): Терминал в проекте
- **Tab 2: "Config"**
  - Терминал в `~/.config`

---

### 2. Layout: my-workspace.kdl (50% / 50%)

**Путь**: `~/.config/zellij/layouts/my-workspace.kdl`

```kdl
layout {
    default_tab_template {
        pane size=1 borderless=true {
            plugin location="zellij:tab-bar"
        }
        children
        pane size=2 borderless=true {
            plugin location="zellij:status-bar"
        }
    }

    tab name="ProjectSnabjenie" focus=true {
        pane split_direction="vertical" {
            pane split_direction="horizontal" size="50%" {
                pane {
                    cwd "/home/mxm/.config"
                }
                pane size="40%" {
                    command "htop"
                }
            }
            pane split_direction="horizontal" size="50%" {
                pane size="70%" {
                    cwd "/mnt/c/Project/ProjectSnabjenie"
                    command "zsh"
                    args "-i" "-c" "n ."
                }
                pane size="30%" {
                    cwd "/mnt/c/Project/ProjectSnabjenie"
                }
            }
        }
    }

    tab name="Config" {
        pane {
            cwd "/home/mxm/.config"
        }
    }
}
```

**Структура**: Идентична первому layout, но с пропорциями 50/50 вместо 40/60.

---

### 3. Скрипт выбора layout

**Путь**: `~/start-zellij-choose.sh`

```bash
#!/bin/bash

# Если уже в zellij - выходим
if [[ -n "$ZELLIJ" ]]; then
    exit 0
fi

echo "Выберите workspace layout:"
echo "1) workspacePrjSnabjenie (40% лево / 60% право) - новый"
echo "2) my-workspace (50% / 50%) - старый"
echo "3) Запустить без layout"
echo ""
read -p "Ваш выбор (1-3): " choice

case $choice in
    1)
        echo "Запускаю workspacePrjSnabjenie..."
        exec zellij --layout "$HOME/.config/zellij/layouts/workspacePrjSnabjenie.kdl"
        ;;
    2)
        echo "Запускаю my-workspace..."
        exec zellij --layout "$HOME/.config/zellij/layouts/my-workspace.kdl"
        ;;
    3)
        echo "Запускаю без layout..."
        exec zellij
        ;;
    *)
        echo "Неверный выбор, запускаю workspacePrjSnabjenie по умолчанию..."
        exec zellij --layout "$HOME/.config/zellij/layouts/workspacePrjSnabjenie.kdl"
        ;;
esac
```

**Назначение**: Интерактивный выбор layout при запуске. Используется автоматически из `.zshrc`.

---

### 4. Скрипт прямого запуска workspacePrjSnabjenie

**Путь**: `~/start-prj-snabjenie.sh`

```bash
#!/bin/bash

# Если уже в zellij - выходим
if [[ -n "$ZELLIJ" ]]; then
    exit 0
fi

# Запускаем новый workspace ProjectSnabjenie (40/60)
exec zellij --layout "$HOME/.config/zellij/layouts/workspacePrjSnabjenie.kdl"
```

**Назначение**: Прямой запуск layout с пропорциями 40/60 без выбора.

---

### 5. Скрипт прямого запуска my-workspace

**Путь**: `~/start-simple.sh`

```bash
#!/bin/bash

# Если уже в zellij - выходим
if [[ -n "$ZELLIJ" ]]; then
    exit 0
fi

# Просто запускаем с layout, без именованной сессии
exec zellij --layout "$HOME/.config/zellij/layouts/my-workspace.kdl"
```

**Назначение**: Прямой запуск layout с пропорциями 50/50 без выбора.

---

### 6. Конфигурация Zsh (автозапуск)

**Путь**: `~/.zshrc` (добавлено в конец файла)

```bash
# Автозапуск Zellij с выбором workspace
if [[ -z "$ZELLIJ" ]]; then
    ~/start-zellij-choose.sh
fi
```

**Назначение**: 
- Проверяет, не запущен ли уже Zellij (переменная `$ZELLIJ`)
- Если не запущен - вызывает скрипт выбора layout
- Предотвращает вложенные запуски Zellij

---

### 7. Конфигурация Alacritty

**Путь**: `C:\Users\z6364\AppData\Roaming\alacritty\alacritty.toml`

**Релевантная секция**:

```toml
[terminal.shell]
program = "wsl"
args = [
    "-d",
    "Ubuntu",
    "--cd",
    "~"
]
```

**Назначение**: 
- Запускает WSL с дистрибутивом Ubuntu
- Начинает с домашней директории (`~`)
- Zsh запускается автоматически через WSL (настройки пользователя)

## 🚀 Способы запуска

### 1. Автоматический (по умолчанию)

Просто откройте **Alacritty** → автоматически появится меню выбора layout.

### 2. Ручной запуск с выбором

```bash
~/start-zellij-choose.sh
```

### 3. Прямой запуск конкретного layout

```bash
# Layout 40/60
~/start-prj-snabjenie.sh

# Layout 50/50
~/start-simple.sh
```

### 4. Запуск без layout

```bash
zellij
```

## ⌨️ Горячие клавиши Zellij

### Управление панелями (`Ctrl+p`)

- `Ctrl+p` → `n` - новая панель
- `Ctrl+p` → стрелки - переключение между панелями
- `Ctrl+p` → `x` - закрыть панель
- `Ctrl+p` → `f` - полноэкранный режим для панели
- `Ctrl+p` → `z` - свернуть/развернуть панель

### Управление табами (`Ctrl+t`)

- `Ctrl+t` → `n` - новый таб
- `Ctrl+t` → `→/←` - переключение между табами
- `Ctrl+t` → `x` - закрыть таб
- `Ctrl+t` → `r` - переименовать таб
- `Ctrl+t` → `[1-9]` - перейти к табу по номеру

### Изменение размера (`Ctrl+n`)

- `Ctrl+n` → стрелки - изменить размер панели
- `Ctrl+n` → `+/-` - увеличить/уменьшить

### Режим поиска (`Ctrl+s`)

- `Ctrl+s` - войти в режим поиска по истории

### Общие

- `Ctrl+q` - выйти из Zellij
- `Ctrl+o` - режим управления сессией
- `Ctrl+g` - выход из любого режима (back to normal)

## 🔧 Настройка и кастомизация

### Изменить автозапуск на конкретный layout

Отредактируйте `~/.zshrc`, замените:

```bash
# Было
if [[ -z "$ZELLIJ" ]]; then
    ~/start-zellij-choose.sh
fi

# Станет (например, всегда 40/60)
if [[ -z "$ZELLIJ" ]]; then
    ~/start-prj-snabjenie.sh
fi
```

### Создать новый layout

1. Создайте файл в `~/.config/zellij/layouts/my-new-layout.kdl`
2. Используйте существующие как шаблон
3. Запустите: `zellij --layout ~/.config/zellij/layouts/my-new-layout.kdl`

### Изменить пропорции панелей

В layout файле измените значения `size`:
- `size="40%"` - 40% от родительской панели
- `size="70%"` - 70% от родительской панели

### Добавить новые команды в панели

```kdl
pane {
    cwd "/path/to/directory"
    command "your-command"
    args "arg1" "arg2"
}
```

## 🐛 Устранение неполадок

### Проблема: Status bar не виден

**Решение**: Убедитесь, что в layout есть `default_tab_template` с `zellij:status-bar`:

```kdl
default_tab_template {
    pane size=1 borderless=true {
        plugin location="zellij:tab-bar"
    }
    children
    pane size=2 borderless=true {
        plugin location="zellij:status-bar"
    }
}
```

### Проблема: Ошибка "There is no active session!"

**Решение**: Проблема обычно в параметре `--session`. Используйте layout без именованных сессий:

```bash
exec zellij --layout "$HOME/.config/zellij/layouts/workspacePrjSnabjenie.kdl"
```

### Проблема: Алиас `n` не работает в layout

**Решение**: Запускайте команду через интерактивный shell:

```kdl
command "zsh"
args "-i" "-c" "n ."
```

### Проблема: Ошибка `^M` в .zshrc

**Решение**: Windows-окончания строк. Конвертируйте в Unix формат:

```bash
sed -i 's/\r$//' ~/.zshrc
```

## 📦 Требования

### Установленные компоненты

- Windows 11
- WSL 2 (Ubuntu)
- Alacritty
- Zellij 0.43.1+
- Zsh
- Neovim (с алиасом `n`)
- htop

### Установка отсутствующих компонентов

```bash
# В WSL Ubuntu
sudo apt update
sudo apt install htop zsh

# Zellij (если не установлен)
cargo install zellij
# или
sudo apt install zellij
```

## 📝 Заметки

1. **Путь к проекту**: `/mnt/c/Project/ProjectSnabjenie` - это Windows путь `C:\Project\ProjectSnabjenie`, доступный через WSL
2. **Домашняя директория**: `/home/mxm` - домашняя директория пользователя в WSL
3. **Алиас `n`**: Предполагается, что в `.zshrc` или `.bash_aliases` настроен алиас `n='nvim'`
4. **Персистентность сессий**: Zellij сессии сохраняются только пока работает WSL. При полном выключении компьютера сессии теряются, но layouts позволяют быстро воссоздать структуру

## 🔄 Backup и восстановление

### Создать backup

```bash
# Backup layouts
tar -czf ~/zellij-backup-$(date +%Y%m%d).tar.gz ~/.config/zellij/layouts/ ~/start-*.sh

# Backup конфигов
cp ~/.zshrc ~/.zshrc.backup
```

### Восстановление

```bash
# Восстановить layouts
tar -xzf ~/zellij-backup-YYYYMMDD.tar.gz -C ~/

# Восстановить .zshrc
cp ~/.zshrc.backup ~/.zshrc
```

## 📚 Дополнительные ресурсы

- [Zellij Documentation](https://zellij.dev/)
- [Zellij Layouts Guide](https://zellij.dev/documentation/layouts.html)
- [Alacritty Configuration](https://alacritty.org/config-alacritty.html)

---

**Дата создания**: 2025-11-06  
**Версия**: 1.0  
**Автор настройки**: AI Assistant with User

